/*
 * Decompiled with CFR 0.152.
 */
package com.tencent.code.intel.util;

import com.tencent.code.intel.manager.LogManager;
import com.tencent.code.intel.util.FuzzyMatchResult;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import org.apache.commons.lang3.StringUtils;

public class FuzzyMatcherUtil {
    private static final double ENOUGH_SIMILARITY_THRESHOLD = 0.95;
    private static final int DEFAULT_TIMEOUT_MS = 1000;
    private static final int DEFAULT_CHAR_DIFF_THRESHOLD = 20;
    private static final double DEFAULT_MIN_LINES_LENGTH_RATIO = 0.5;
    private static final double DEFAULT_MAX_LINES_LENGTH_RATIO = 1.5;
    private static final double DEFAULT_SIMILARITY_LINES_LENGTH_RATIO = 0.3;
    private static final int DEFAULT_LINE_DIFF_THRESHOLD = 2;

    public static List<FuzzyMatchResult> find(String longStr, String pattern) {
        try {
            return FuzzyMatcherUtil.find(longStr, pattern, 0.9, 1000, false);
        }
        catch (Exception e) {
            LogManager.warn(FuzzyMatcherUtil.class, "Fuzzy matching error: " + e.getMessage());
            return Collections.emptyList();
        }
    }

    public static List<FuzzyMatchResult> find(String longStr, String pattern, double threshold, int timeoutMs, boolean onlyExactMatch) {
        String normalizedPattern;
        if (pattern == null || pattern.isEmpty() || longStr == null || longStr.isEmpty()) {
            return Collections.emptyList();
        }
        long startTime = System.currentTimeMillis();
        String normalizedLongStr = FuzzyMatcherUtil.normalizeString(longStr);
        FuzzyMatchResult exactMatch = FuzzyMatcherUtil.findExactMatch(normalizedLongStr, normalizedPattern = FuzzyMatcherUtil.normalizeString(pattern), longStr);
        if (exactMatch != null) {
            return Collections.singletonList(exactMatch);
        }
        LogManager.warn(FuzzyMatcherUtil.class, "Can not find exactMatch, using fuzzy matching");
        if (onlyExactMatch) {
            return Collections.emptyList();
        }
        ArrayList<FuzzyMatchResult> results = new ArrayList<FuzzyMatchResult>();
        int minLength = normalizedPattern.length() - 20;
        int maxLength = normalizedPattern.length() + 20;
        String[] originalLines = longStr.split("\\r?\\n", -1);
        String[] longStrLines = normalizedLongStr.split("\n", -1);
        String[] patternLines = normalizedPattern.split("\n", -1);
        for (int startLine = 0; startLine < longStrLines.length; ++startLine) {
            if (System.currentTimeMillis() - startTime > (long)timeoutMs) {
                LogManager.warn(FuzzyMatcherUtil.class, "Fuzzy matching timeout reached, returning current results");
                break;
            }
            if (startLine + patternLines.length > longStrLines.length) break;
            String firstPatternLine = patternLines[0];
            String currentLine = longStrLines[startLine].trim();
            if ((double)currentLine.length() < (double)firstPatternLine.length() * 0.5 || (double)currentLine.length() > (double)firstPatternLine.length() * 1.5) continue;
            int minWindowLines = Math.max(1, patternLines.length - 2);
            int maxWindowLines = patternLines.length + 2;
            for (int windowSize = minWindowLines; windowSize <= maxWindowLines && startLine + windowSize <= longStrLines.length; ++windowSize) {
                double similarity;
                CharSequence[] windowLines = Arrays.copyOfRange(longStrLines, startLine, startLine + windowSize);
                String windowText = String.join((CharSequence)"\n", windowLines);
                if (windowText.length() < minLength || windowText.length() > maxLength || (similarity = FuzzyMatcherUtil.calculateSimilarity(windowText, normalizedPattern)) <= threshold) continue;
                int startPos = 0;
                int emptyLines = originalLines[0].isEmpty() ? 1 : 0;
                for (int i = 0; i < startLine + emptyLines; ++i) {
                    startPos += longStr.substring(startPos += originalLines[i].length()).startsWith("\r\n") ? 2 : 1;
                    if (i + 1 >= originalLines.length || !StringUtils.isBlank((CharSequence)originalLines[i + 1])) continue;
                    ++emptyLines;
                }
                int newStartLineIndex = startLine + emptyLines;
                int endLineIndex = startLine + windowSize - 1;
                int endPos = FuzzyMatcherUtil.calcEndPosition(longStr, originalLines, startPos, startLine, endLineIndex, emptyLines);
                FuzzyMatchResult match = new FuzzyMatchResult(true, startPos, longStr.substring(startPos, endPos), similarity, newStartLineIndex, endLineIndex + emptyLines);
                results.add(match);
            }
        }
        int patternLength = pattern.split("\\r?\\n", -1).length;
        return results.stream().sorted((a, b) -> {
            int bLengthDiff;
            double similarityDiff = b.getSimilarity() - a.getSimilarity();
            if (Math.abs(similarityDiff) > 0.01) {
                return Double.compare(b.getSimilarity(), a.getSimilarity());
            }
            int aLengthDiff = Math.abs(a.getMatchedText().split("\\r?\\n", -1).length - patternLength);
            if (Math.abs(aLengthDiff - (bLengthDiff = Math.abs(b.getMatchedText().split("\\r?\\n", -1).length - patternLength))) > 2) {
                return Integer.compare(aLengthDiff, bLengthDiff);
            }
            return Double.compare(b.getSimilarity(), a.getSimilarity());
        }).limit(1L).collect(Collectors.toList());
    }

    private static String normalizeString(String str) {
        return str.replaceAll("\\r\\n", "\n").replaceAll("(\\n)+", "\n").replaceAll("\\t", " ").replaceAll(" +", " ").replaceAll("\n\\s+", "\n").replaceAll("\\s+\n", "\n").trim();
    }

    private static double calculateSimilarity(String str1, String str2) {
        double lenRatio = (double)Math.min(str1.length(), str2.length()) / (double)Math.max(str1.length(), str2.length());
        if (lenRatio < 0.7) {
            return 0.0;
        }
        String clean1 = str1.replaceAll("\\s+", "");
        String clean2 = str2.replaceAll("\\s+", "");
        if ((double)Math.abs(clean1.length() - clean2.length()) > (double)clean1.length() * 0.3) {
            return 0.0;
        }
        HashMap<Character, Integer> charFreq1 = new HashMap<Character, Integer>();
        HashMap<Character, Integer> charFreq2 = new HashMap<Character, Integer>();
        String key1 = clean1.replaceAll("[^a-zA-Z0-9]", "");
        String key2 = clean2.replaceAll("[^a-zA-Z0-9]", "");
        for (char c : key1.toCharArray()) {
            charFreq1.merge(Character.valueOf(c), 1, Integer::sum);
        }
        for (char c : key2.toCharArray()) {
            charFreq2.merge(Character.valueOf(c), 1, Integer::sum);
        }
        double freqSimilarity = 0.0;
        int totalChars = 0;
        for (Map.Entry entry : charFreq1.entrySet()) {
            char c = ((Character)entry.getKey()).charValue();
            int count1 = (Integer)entry.getValue();
            int count2 = charFreq2.getOrDefault(Character.valueOf(c), 0);
            freqSimilarity += (double)Math.min(count1, count2);
            totalChars += Math.max(count1, count2);
        }
        for (Map.Entry entry : charFreq2.entrySet()) {
            if (charFreq1.containsKey(entry.getKey())) continue;
            totalChars += ((Integer)entry.getValue()).intValue();
        }
        double freqRatio = freqSimilarity / (double)totalChars;
        if (freqRatio < 0.6) {
            return freqRatio;
        }
        int lcsLength = FuzzyMatcherUtil.longestCommonSubsequence(clean1, clean2);
        return (double)lcsLength * 2.0 / (double)(clean1.length() + clean2.length());
    }

    public static int longestCommonSubsequence(String str1, String str2) {
        int m = str1.length();
        int n = str2.length();
        int[] prev = new int[n + 1];
        int[] curr = new int[n + 1];
        for (int i = 1; i <= m; ++i) {
            for (int j = 1; j <= n; ++j) {
                curr[j] = str1.charAt(i - 1) == str2.charAt(j - 1) ? prev[j - 1] + 1 : Math.max(prev[j], curr[j - 1]);
            }
            int[] temp = prev;
            prev = curr;
            curr = temp;
        }
        return prev[n];
    }

    private static FuzzyMatchResult findExactMatch(String normalizedLongStr, String normalizedPattern, String originalLongStr) {
        int position = normalizedLongStr.indexOf(normalizedPattern);
        if (position != -1) {
            String beforeText = normalizedLongStr.substring(0, position);
            int startLineIndex = beforeText.split("\n", -1).length - 1;
            int matchEndPosition = position + normalizedPattern.length();
            int endLineIndex = normalizedLongStr.substring(0, matchEndPosition).split("\n", -1).length - 1;
            String[] originalLines = originalLongStr.split("\\r?\\n", -1);
            int startPos = 0;
            int emptyLines = originalLines[0].isEmpty() ? 1 : 0;
            for (int i = 0; i < startLineIndex + emptyLines; ++i) {
                startPos += originalLongStr.substring(startPos += originalLines[i].length()).startsWith("\r\n") ? 2 : 1;
                if (i + 1 >= originalLines.length || !StringUtils.isBlank((CharSequence)originalLines[i + 1])) continue;
                ++emptyLines;
            }
            int newStartLineIndex = startLineIndex + emptyLines;
            int endPos = FuzzyMatcherUtil.calcEndPosition(originalLongStr, originalLines, startPos, startLineIndex, endLineIndex, emptyLines);
            return new FuzzyMatchResult(true, startPos, originalLongStr.substring(startPos, endPos), 1.0, newStartLineIndex, endLineIndex + emptyLines);
        }
        return null;
    }

    public static int calcEndPosition(String originalLongStr, String[] originalLines, int startPos, int startLineIndex, int endLineIndex, int emptyLines) {
        int newStartLineIndex;
        int resultPos = startPos;
        for (int i = newStartLineIndex = startLineIndex + emptyLines; i <= endLineIndex + emptyLines && i < originalLines.length; ++i) {
            String remainingText;
            if (i + 1 <= endLineIndex + emptyLines && StringUtils.isBlank((CharSequence)originalLines[i + 1])) {
                ++emptyLines;
            }
            if ((remainingText = originalLongStr.substring(resultPos += originalLines[i].length())).isEmpty() || i + 1 > endLineIndex + emptyLines) continue;
            resultPos += remainingText.startsWith("\r\n") ? 2 : 1;
        }
        return resultPos;
    }
}

